package edu.uky.ai.lp.ai;

import java.util.LinkedList;
import java.util.Queue;
import java.util.Set;

import edu.uky.ai.lp.Action;

/**
 * Utility methods for navigating paths though the Wumpus World.
 * 
 * @author Stephen G. Ware
 */
public class Path {
	
	/**
	 * Returns the file index of a square.
	 * 
	 * @param location a square, as a string
	 * @return the file (0 to 3)
	 */
	private static final int file(String location) {
		switch(location.charAt(0)) {
		case 'a': return 0;
		case 'b': return 1;
		case 'c': return 2;
		default: return 3;
		}
	}
	
	/**
	 * Returns the rank index of a square.
	 * 
	 * @param location a square, as a string
	 * @return the rank (0 to 3)
	 */
	private static final int rank(String location) {
		switch(location.charAt(1)) {
		case '1': return 0;
		case '2': return 1;
		case '3': return 2;
		default: return 3;
		}
	}
	
	/**
	 * Returns the location of a square based on its file and rank.
	 * 
	 * @param file the file of a square (0 to 3)
	 * @param rank the rank of a square (0 to 3)
	 * @return the location as a string
	 */
	private static final String location(int file, int rank) {
		String location = "";
		switch(file) {
		case 0: location += "a"; break;
		case 1: location += "b"; break;
		case 2: location += "c"; break;
		case 3: location += "d"; break;
		}
		return location += Integer.toString(rank + 1);
	}
	
	/**
	 * Given the player's current location and intended destination, return the
	 * next move action that needs to be taken to get to the destination.  Only
	 * visited squares will be considered.
	 * 
	 * @param from the player's current location
	 * @param to the player's destination
	 * @param visited the set of visited squares
	 * @return the next move action to take to reach the destination
	 */
	public static final Action path(String from, String to, Set<String> visited) {
		Queue<LinkedList<String>> queue = new LinkedList<>();
		LinkedList<String> path = new LinkedList<>();
		path.add(from);
		queue.add(path);
		while(!queue.isEmpty()) {
			path = queue.poll();
			if(path.getLast().equals(to))
				break;
			consider(queue, path, Action.UP, to, visited);
			consider(queue, path, Action.DOWN, to, visited);
			consider(queue, path, Action.RIGHT, to, visited);
			consider(queue, path, Action.LEFT, to, visited);
		}
		if(getNewLocation(from, Action.UP).equals(path.get(1)))
			return Action.UP;
		else if(getNewLocation(from, Action.DOWN).equals(path.get(1)))
			return Action.DOWN;
		else if(getNewLocation(from, Action.RIGHT).equals(path.get(1)))
			return Action.RIGHT;
		else if(getNewLocation(from, Action.LEFT).equals(path.get(1)))
			return Action.LEFT;
		else
			return null;
	}
	
	@SuppressWarnings("unchecked")
	private static final void consider(Queue<LinkedList<String>> queue, LinkedList<String> path, Action action, String goal, Set<String> visited) {
		String next = getNewLocation(path.getLast(), action);
		if(visited.contains(next) || next.equals(goal)) {
			path = (LinkedList<String>) path.clone();
			path.add(next);
			queue.add(path);
		}
	}
	
	/**
	 * Given the player's current location and a next move action to take,
	 * return the player's new location after the action is taken.
	 * 
	 * @param current the player's current location
	 * @param action the next move action to take
	 * @return the player's location after the action
	 */
	public static final String getNewLocation(String current, Action action) {
		int file = file(current);
		int rank = rank(current);
		if(action == Action.UP && rank < 3)
			return location(file, rank + 1);
		else if(action == Action.DOWN && rank > 0)
			return location(file, rank - 1);
		else if(action == Action.RIGHT && file < 3)
			return location(file + 1, rank);
		else if(action == Action.LEFT && file > 0)
			return location(file - 1, rank);
		else
			return current;
	}
}
